/*
 * routines for making a copy of a whole fabric network
 */
#include <stdio.h>
#include <string.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"

/*
 * Make a clone copy of one xcvr
 */
struct lf_xcvr *
lf_clone_xcvr(
  struct lf_xcvr *oxcp)
{
  struct lf_xcvr *xcp;

  /* allocate a new xcvr struct */
  LF_CALLOC(xcp, struct lf_xcvr, 1);

  /* copy over what we can */
  xcp->ln_type = oxcp->ln_type;
  xcp->port = oxcp->port;
  
  /* duplicate the port array */
  xcp->num_ports = oxcp->num_ports;
  xcp->num_conns = oxcp->num_conns;
  LF_CALLOC(xcp->ports, union lf_node *, xcp->num_ports);
  LF_CALLOC(xcp->rports, int, xcp->num_ports);

  return xcp;

 except:
  lf_free_xcvr(xcp);
  return NULL;
}

/*
 * Make a clone copy of one NIC
 */
struct lf_nic *
lf_clone_nic(
  struct lf_nic *onicp)
{
  struct lf_nic *nicp;
  int i;

  /* allocate a new nic struct */
  nicp = lf_alloc_nic(onicp->num_ports);
  if (nicp == NULL) LF_ERROR(("Error allocating clone NIC"));

  /* copy over the mac addr, serial number, etc. */
  nicp->slot = onicp->slot;
  nicp->host_nic_id = onicp->host_nic_id;
  nicp->partition = onicp->partition;
  LF_MAC_COPY(nicp->mac_addr, onicp->mac_addr);
  LF_DUP_STRING(nicp->serial_no, onicp->serial_no);
  LF_DUP_STRING(nicp->product_id, onicp->product_id);
  nicp->def = onicp->def;
  
  /* copy NIC xcvrs */
  for (i=0; i<nicp->num_ports; ++i) {
    struct lf_xcvr *xcp;

    xcp = LF_XCVR(onicp->phys_ports[i]);
    if (xcp != NULL && xcp->ln_type == LF_NODE_NIC_XCVR) {
      xcp = lf_clone_xcvr(xcp);
      if (xcp == NULL) LF_ERROR(("Error cloning xcvr"));

      /* save xcvr pointer and backpointer to NIC */
      /* only 1 port per xcvr supported just now... */
      nicp->phys_ports[i] = LF_NODE(xcp);
      nicp->phys_rports[i] = xcp->num_conns + 0;
      xcp->p.nic = nicp;

      /* connect xcvr back to NIC */
      xcp->ports[xcp->num_conns + 0] = LF_NODE(nicp);
      xcp->rports[xcp->num_conns + 0] = i;
    }
  }

  return nicp;

 except:
  if (nicp != NULL) lf_free_nic(nicp);
  return NULL;
}

/*
 * Make a clone copy of one host
 */
struct lf_host *
lf_clone_host(
  struct lf_host *ohp)
{
  struct lf_host *hp;
  int i;

  /* allocate a new host struct */
  LF_CALLOC(hp, struct lf_host, 1);

  /* copy hostname */
  LF_DUP_STRING(hp->hostname, ohp->hostname);
  hp->fw_type = ohp->fw_type;
  hp->fma_flags = ohp->fma_flags;

  /* copy over the NICs */
  hp->num_nics = ohp->num_nics;
  LF_CALLOC(hp->nics, struct lf_nic *, hp->num_nics);
  for (i=0; i<hp->num_nics; ++i) {
    struct lf_nic *nicp;

    nicp = lf_clone_nic(ohp->nics[i]);
    if (nicp == NULL) LF_ERROR(("Error cloning NIC"));

    /* save NIC pointer and backpointer to host */
    hp->nics[i] = nicp;
    nicp->host = hp;
  }

  return hp;

 except:
  lf_free_host(hp);
  return NULL;
}

/*
 * Make a clone copy of one xbar
 */
struct lf_xbar *
lf_clone_xbar(
  struct lf_xbar *oxp)
{
  struct lf_xbar *xp;

  /* allocate a new xbar struct */
  xp = lf_alloc_xbar(oxp->num_ports);
  if (xp == NULL) LF_ERROR(("Error allocating clone xbar"));

  /* copy over the mac addr, serial number, etc. */
  xp->xbar_no = oxp->xbar_no;
  xp->xbar_id = oxp->xbar_id;
  
  return xp;

 except:
  if (xp != NULL) lf_free_xbar(xp);
  return NULL;
}

/*
 * Make a clone copy of one linecard
 */
struct lf_linecard *
lf_clone_linecard(
  struct lf_enclosure *ep,
  struct lf_linecard *olp)
{
  struct lf_linecard *lp;
  int i;

  /* allocate a new linecard struct */
  LF_CALLOC(lp, struct lf_linecard, 1);

  /* copy serial number, etc. */
  LF_DUP_STRING(lp->serial_no, olp->serial_no);
  LF_DUP_STRING(lp->product_id, olp->product_id);
  lp->slot = olp->slot;
  lp->backplane = olp->backplane;
  lp->def = olp->def;

  /* copy over the xbars */
  lp->num_xbars = olp->num_xbars;
  LF_CALLOC(lp->xbars, union lf_node *, lp->num_xbars);
  for (i=0; i<lp->num_xbars; ++i) {
    struct lf_xbar *xp;

    xp = lf_clone_xbar(LF_XBAR(olp->xbars[i]));
    if (xp == NULL) LF_ERROR(("Error cloning xbar"));

    /* save xbar pointer and backpointer to linecard */
    lp->xbars[i] = LF_NODE(xp);
    xp->linecard = lp;
  }

  /* copy over the xcvrs */
  lp->num_xcvrs = olp->num_xcvrs;
  LF_CALLOC(lp->xcvrs, union lf_node *, lp->num_xcvrs);
  LF_CALLOC(lp->xcvr_labels, int, lp->num_xcvrs);
  for (i=0; i<lp->num_xcvrs; ++i) {
    struct lf_xcvr *xcp;

    xcp = lf_clone_xcvr(LF_XCVR(olp->xcvrs[i]));
    if (xcp == NULL) LF_ERROR(("Error cloning xcvr"));

    /* save xcvr pointer and backpointer to linecard */
    lp->xcvrs[i] = LF_NODE(xcp);
    xcp->p.linecard = lp;

    lp->xcvr_labels[i] = olp->xcvr_labels[i];
  }

  /* make the internal connections */
  lf_make_linecard_internal_links(lp);

  /* save linecard pointer and backpointer to enclosure */
  ep->slots[olp->slot] = lp;
  lp->enclosure = ep;

  return lp;

 except:
  lf_free_linecard(lp);
  return NULL;
}

/*
 * Make a clone copy of one enclosure
 */
struct lf_enclosure *
lf_clone_enclosure(
  struct lf_enclosure *oep)
{
  struct lf_enclosure *ep;
  int i;

  /* allocate a new enclosure struct */
  LF_CALLOC(ep, struct lf_enclosure, 1);

  /* copy name */
  LF_DUP_STRING(ep->name, oep->name);
  LF_DUP_STRING(ep->product_id, oep->product_id);
  ep->def = oep->def;
  ep->num_lc_slots = oep->num_lc_slots;
  ep->lc_slotbase = oep->lc_slotbase;
  ep->num_bp_slots = oep->num_bp_slots;
  ep->bp_slotbase = oep->bp_slotbase;

  /* copy over the linecards */
  ep->num_slots = oep->num_slots;
  LF_CALLOC(ep->slots, struct lf_linecard *, ep->num_slots);
  for (i=0; i<ep->num_slots; ++i) {
    struct lf_linecard *lp;

    if (oep->slots[i] == NULL) continue;

    lp = lf_clone_linecard(ep, oep->slots[i]);
    if (lp == NULL) LF_ERROR(("Error cloning linecard"));
  }

  /* make all the internal links */
  lf_create_implicit_links(ep);

  return ep;

 except:
  lf_free_enclosure(ep);
  return NULL;
}

union lf_node *
lf_find_clone_counterpart(
  union lf_node *onp,
  struct lf_fabric *fp)
{
  switch (onp->ln_type) {
    case LF_NODE_XBAR:
      {
	struct lf_xbar *oxp;
	struct lf_enclosure *ep;
	int l;
	int x;

	oxp = LF_XBAR(onp);
	ep = lf_find_enclosure_by_name(fp, oxp->linecard->enclosure->name);
	if (ep == NULL) LF_ERROR(("Cannot find enclosure %s in fabric",
	    oxp->linecard->enclosure->name));

	l = oxp->linecard->slot;
	x = oxp->xbar_no;
	return ep->slots[l]->xbars[x];
      }
      break;

    case LF_NODE_LC_XCVR:
      {
	struct lf_xcvr *oxcp;
	struct lf_enclosure *ep;
	int l;
	int x;

	oxcp = LF_XCVR(onp);
	ep = lf_find_enclosure_by_name(fp, oxcp->p.linecard->enclosure->name);
	if (ep == NULL) LF_ERROR(("Cannot find enclosure %s in fabric",
	    oxcp->p.linecard->enclosure->name));

	l = oxcp->p.linecard->slot;
	x = oxcp->port;
	return ep->slots[l]->xcvrs[x];
      }
      break;

    case LF_NODE_NIC_XCVR:
      {
	struct lf_xcvr *oxcp;
	struct lf_host *hp;
	int n;
	int p;

	oxcp = LF_XCVR(onp);
	hp = lf_find_host_by_name(fp, oxcp->p.nic->host->hostname);
	if (hp == NULL) LF_ERROR(("Cannot find host %s in fabric",
	    oxcp->p.nic->host->hostname));

	n = oxcp->p.nic->slot;
	p = oxcp->port;
	return hp->nics[n]->phys_ports[p];
      }
      break;

    case LF_NODE_NIC:
      {
	struct lf_nic *onicp;
	struct lf_host *hp;
	int n;

	onicp = LF_NIC(onp);
	hp = lf_find_host_by_name(fp, onicp->host->hostname);
	if (hp == NULL) LF_ERROR(("Cannot find host %s in fabric",
	    onicp->host->hostname));

	n = onicp->slot;
	return LF_NODE(hp->nics[n]);
      }
      break;
  }

 except:
  return NULL;
}

/*
 * Clone all the links in one fabric to another.  Scan through all xbars and
 * all linecard xcvrs and clone each link.
 */
int
lf_clone_links(
  struct lf_fabric *ofp,
  struct lf_fabric *fp)
{
  int e;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    struct lf_enclosure *oep;
    int l;

    ep = fp->enclosures[e];
    oep = ofp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      struct lf_linecard *olp;
      int x;

      lp = ep->slots[l];
      olp = oep->slots[l];
      if (lp == NULL) continue;

      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;
	struct lf_xbar *oxp;
	int p;

	xp = LF_XBAR(lp->xbars[x]);
	oxp = LF_XBAR(olp->xbars[x]);
	for (p=0; p<xp->num_ports; ++p) {
	  union lf_node *np;
	  union lf_node *onp;

	  np = xp->topo_ports[p];
	  onp = oxp->topo_ports[p];
	  if (onp != NULL && np == NULL) {
	    np = lf_find_clone_counterpart(onp, fp);
	    if (np == NULL) LF_ERROR(("Error finding topo clone counterpart"));

	    lf_make_topo_link(LF_NODE(xp), p, np, oxp->topo_rports[p]);
	    lf_make_topo_link(np, oxp->topo_rports[p], LF_NODE(xp), p);
	  }

	  np = xp->phys_ports[p];
	  onp = oxp->phys_ports[p];
	  if (onp != NULL && np == NULL) {
	    np = lf_find_clone_counterpart(onp, fp);
	    if (np == NULL) LF_ERROR(("Error finding phys clone counterpart"));

	    lf_make_phys_link(LF_NODE(xp), p, np, oxp->phys_rports[p]);
	    lf_make_phys_link(np, oxp->phys_rports[p], LF_NODE(xp), p);
	  }
	}
      }

      for (x=0; x<lp->num_xcvrs; ++x) {
	struct lf_xcvr *xcp;
	struct lf_xcvr *oxcp;
	int p;

	xcp = LF_XCVR(lp->xcvrs[x]);
	oxcp = LF_XCVR(olp->xcvrs[x]);
	for (p=0; p<xcp->num_ports; ++p) {
	  union lf_node *np;
	  union lf_node *onp;

	  np = xcp->ports[p];
	  onp = oxcp->ports[p];
	  if (onp != NULL && np == NULL) {
	    np = lf_find_clone_counterpart(onp, fp);
	    if (np == NULL) LF_ERROR(("Error finding xcvr clone counterpart"));

	    lf_make_phys_link(LF_NODE(xcp), p, np, oxcp->rports[p]);
	    lf_make_phys_link(np, oxcp->rports[p], LF_NODE(xcp), p);
	  }
	}
      }
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Clone all the links for one linecard in one fabric to another. 
 * This assumes lp is a cloned copy of clp
 */
int
lf_clone_linecard_links(
  struct lf_fabric *fp,
  struct lf_linecard *lp,
  struct lf_fabric *cfp,
  struct lf_linecard *clp)
{
  /* XXX */
  return -1;
}

/*
 * Clone all the links for one host in one fabric to another. 
 * This assumes nhp is a cloned copy of ohp
 */
int
lf_clone_host_links(
  struct lf_fabric *ofp,
  struct lf_host *ohp,
  struct lf_fabric *nfp,
  struct lf_host *nhp)
{
  int n;
  int rc;

  for (n=0; n<ohp->num_nics; ++n) {
    rc = lf_clone_nic_links(ofp, ohp->nics[n], nfp, nhp->nics[n]);
    if (rc != 0) {
      LF_ERROR(("Error cloning links for NIC %d of host %s", n, ohp->hostname));
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Clone all the links for one NIC to a NIC in another fabric
 * nnicp is the clones copy of onicp
 */
int
lf_clone_nic_links(
  struct lf_fabric *ofp,
  struct lf_nic *onicp,
  struct lf_fabric *nfp,
  struct lf_nic *nnicp)
{
  int p;

  for (p=0; p<nnicp->num_ports; ++p) {

    union lf_node *onp;
    union lf_node *nnp;
    struct lf_xcvr *oxcp;
    struct lf_xcvr *nxcp;

    /* Make the topo links */
    onp = onicp->topo_ports[p];
    nnp = nnicp->topo_ports[p];

    if (onp != NULL && nnp == NULL) {
      nnp = lf_find_clone_counterpart(onp, nfp);
      if (nnp == NULL) LF_ERROR(("Error finding topo clone counterpart"));

      lf_make_topo_link(LF_NODE(nnicp), p, nnp, onicp->topo_rports[p]);
      lf_make_topo_link(nnp, onicp->topo_rports[p], LF_NODE(nnicp), p);
    }

    /* Make phys links */
    oxcp = LF_XCVR(onicp->phys_ports[p]);
    nxcp = LF_XCVR(nnicp->phys_ports[p]);

    /* We assume only single port xcvrs */
    onp = oxcp->ports[0];
    nnp = nxcp->ports[0];

    if (onp != NULL && nnp == NULL) {
      nnp = lf_find_clone_counterpart(onp, nfp);
      if (nnp == NULL) LF_ERROR(("Error finding phys clone counterpart"));

      lf_make_phys_link(LF_NODE(nxcp), 0, nnp, oxcp->rports[0]);
      lf_make_phys_link(nnp, oxcp->rports[0], LF_NODE(nxcp), 0);
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Make a clone copy of a fabric
 */
struct lf_fabric *
lf_clone_fabric(
  struct lf_fabric *ofp)
{
  struct lf_fabric *fp;
  int i;
  int rc;

  /* allocate holders */
  LF_CALLOC(fp, struct lf_fabric, 1);

  /* clone all the hosts */
  fp->num_hosts = ofp->num_hosts;
  LF_CALLOC(fp->hosts, struct lf_host *, fp->num_hosts);
  for (i=0; i<fp->num_hosts; ++i) {
    fp->hosts[i] = lf_clone_host(ofp->hosts[i]);
    if (fp->hosts[i] == NULL) LF_ERROR(("Error cloning host"));
  }

  /* clone all the enclosures */
  fp->num_enclosures = ofp->num_enclosures;
  LF_CALLOC(fp->enclosures, struct lf_enclosure *, fp->num_enclosures);
  for (i=0; i<fp->num_enclosures; ++i) {
    fp->enclosures[i] = lf_clone_enclosure(ofp->enclosures[i]);
    if (fp->enclosures[i] == NULL) LF_ERROR(("Error cloning enclosure"));
  }

  /* now that all the elements are in place, clone all the links */
  rc = lf_clone_links(ofp, fp);
  if (rc == -1) LF_ERROR(("Error cloning links"));

  return fp;

 except:
  lf_free_fabric(fp);
  return NULL;
}
